package de.ugoe.cs.crosspare;

import java.io.FileWriter;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.security.InvalidParameterException;

public class ConfigurationBuilder {
    
    private static enum Dataset {MDP, JURECZKO, FILTERJURECZKO, AEEEM, RELINK, NETGENE, SELECTEDJURECZKO, AEEEM_LDHH, AEEEM_WCHU, AEEEM_LDHHWCHU, SMARTSHARK_ALL, SMARTSHARK_AST, SMARTSHARK_SM}
    
    private static final String storageFolder = "config/";
    
    public static void main(String[] args) {
        for( Dataset dataset : Dataset.values() ) {
            // baselines
            writeFile("ALL", dataset);
            writeFile("CV", dataset);
            writeFile("Random", dataset);
            writeFile("Trivial", dataset);
            // publications
            writeFile("Koshgoftaar08", dataset);
            writeFile("Watanabe08", dataset);
            writeFile("Turhan09", dataset);
            writeFile("Zimmermann09", dataset);
            writeFile("CamargoCruz09", dataset);
            writeFile("Liu10", dataset);
            writeFile("Menzies11", dataset);
            writeFile("Ma12", dataset);
            writeFile("Peters12", dataset);
            writeFile("Uchigaki12", dataset);
            writeFile("Canfora13", dataset);
            writeFile("Peters13", dataset);
            writeFile("Herbold13", dataset);
            writeFile("ZHe13", dataset);
            writeFile("Nam13", dataset);
            writeFile("Panichella14", dataset);
            writeFile("Ryu14", dataset);
            writeFile("PHe15", dataset);
            writeFile("Peters15", dataset);
            writeFile("Kawata15", dataset);
            writeFile("YZhang15", dataset);
            writeFile("Amasaki15", dataset);
            writeFile("Ryu15", dataset);
            writeFile("Nam15", dataset);
        }
    }
    
    public static void writeFile(String approach, Dataset dataset) {
        try(FileWriter writer = new FileWriter(storageFolder + dataset.toString() + "-" + approach + ".xml");) {
            writer.append((String) ConfigurationBuilder.class.getMethod(approach, Dataset.class).invoke(null, dataset));
            writer.flush();
        }
        catch (IOException | IllegalAccessException | IllegalArgumentException | InvocationTargetException | NoSuchMethodException | SecurityException e) {
            e.printStackTrace();
        }
    }
    
    public static void preamble(StringBuilder configFile) {
        configFile.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
        configFile.append("<config xmlns=\"experimentconfig\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"experimentconfig experimentconfig.xsd\">\n");
    }
    
    public static void postamble(StringBuilder configFile) {
        configFile.append(" <storage name=\"MySQLResultStorage\" param=\"\" />\n");
        configFile.append("</config>");
    }
    
    public static void trainers(StringBuilder configFile) {
        configFile.append(" <trainer name=\"WekaTraining\" param=\"NB weka.classifiers.bayes.NaiveBayes\" />\n");
        configFile.append(" <trainer name=\"WekaTraining\" param=\"RF weka.classifiers.trees.RandomForest -CVPARAM I 5 25 5\" />\n");
        configFile.append(" <trainer name=\"WekaTraining\" param=\"DT weka.classifiers.trees.J48 -CVPARAM C 0.1 0.3 5\" />\n");
        configFile.append(" <trainer name=\"WekaTraining\" param=\"LR weka.classifiers.functions.Logistic\" />\n");
        configFile.append(" <trainer name=\"WekaTraining\" param=\"NET weka.classifiers.functions.RBFNetwork -CVPARAM W 0.1 10.0 3.0 L 2.0 18.0 3.0\" />\n");
        configFile.append(" <trainer name=\"WekaTraining\" param=\"SVM weka.classifiers.functions.SMO -K weka.classifiers.functions.supportVector.RBFKernel\" />\n");
    }
    
    public static void trainersBagging(StringBuilder configFile) {
        configFile.append(" <setwisetrainer name=\"WekaBaggingTraining\" param=\"NB weka.classifiers.bayes.NaiveBayes\" />\n");
        configFile.append(" <setwisetrainer name=\"WekaBaggingTraining\" param=\"RF weka.classifiers.trees.RandomForest -CVPARAM I 5 25 5\" />\n");
        configFile.append(" <setwisetrainer name=\"WekaBaggingTraining\" param=\"DT weka.classifiers.trees.J48 -CVPARAM C 0.1 0.3 5\" />\n");
        configFile.append(" <setwisetrainer name=\"WekaBaggingTraining\" param=\"LR weka.classifiers.functions.Logistic\" />\n");
        configFile.append(" <setwisetrainer name=\"WekaBaggingTraining\" param=\"NET weka.classifiers.functions.RBFNetwork -CVPARAM W 0.1 10.0 3.0 L 2.0 18.0 3.0\" />\n");
        configFile.append(" <setwisetrainer name=\"WekaBaggingTraining\" param=\"SVM weka.classifiers.functions.SMO -K weka.classifiers.functions.supportVector.RBFKernel\" />\n");
    }
    
    public static void trainersLocalWhere(StringBuilder configFile) {
        configFile.append(" <trainer name=\"WekaLocalFQTraining\" param=\"NB weka.classifiers.bayes.NaiveBayes\" />\n");
        configFile.append(" <trainer name=\"WekaLocalFQTraining\" param=\"RF weka.classifiers.trees.RandomForest -CVPARAM I 5 25 5\" />\n");
        configFile.append(" <trainer name=\"WekaLocalFQTraining\" param=\"DT weka.classifiers.trees.J48 -CVPARAM C 0.1 0.3 5\" />\n");
        configFile.append(" <trainer name=\"WekaLocalFQTraining\" param=\"LR weka.classifiers.functions.Logistic\" />\n");
        configFile.append(" <trainer name=\"WekaLocalFQTraining\" param=\"NET weka.classifiers.functions.RBFNetwork -CVPARAM W 0.1 10.0 3.0 L 2.0 18.0 3.0\" />\n");
        configFile.append(" <trainer name=\"WekaLocalFQTraining\" param=\"SVM weka.classifiers.functions.SMO -K weka.classifiers.functions.supportVector.RBFKernel\" />\n");
        configFile.append(" <trainer name=\"WekaLocalFQTraining\" param=\"WHICH de.ugoe.cs.cpdp.wekaclassifier.WHICH\" />\n");
    }
    
    public static void trainersLASER(StringBuilder configFile) {
        configFile.append(" <trainer name=\"WekaLASERTraining\" param=\"NB weka.classifiers.bayes.NaiveBayes\" />\n");
        configFile.append(" <trainer name=\"WekaLASERTraining\" param=\"RF weka.classifiers.trees.RandomForest -CVPARAM I 5 25 5\" />\n");
        configFile.append(" <trainer name=\"WekaLASERTraining\" param=\"DT weka.classifiers.trees.J48 -CVPARAM C 0.1 0.3 5\" />\n");
        configFile.append(" <trainer name=\"WekaLASERTraining\" param=\"LR weka.classifiers.functions.Logistic\" />\n");
        configFile.append(" <trainer name=\"WekaLASERTraining\" param=\"NET weka.classifiers.functions.RBFNetwork -CVPARAM W 0.1 10.0 3.0 L 2.0 18.0 3.0\" />\n");
        configFile.append(" <trainer name=\"WekaLASERTraining\" param=\"SVM weka.classifiers.functions.SMO -K weka.classifiers.functions.supportVector.RBFKernel\" />\n");
    }
        
    public static void dataset(StringBuilder configFile, Dataset dataset) {
        switch (dataset)
        {
            case MDP:
                configFile.append(" <loader name=\"NasaARFFFolderLoader\" datalocation=\"benchmark/data/MDP\" relative=\"false\"/>\n");
                break;
            case JURECZKO:
                configFile.append(" <loader name=\"CSVFolderLoader\" datalocation=\"benchmark/data/JURECZKO\" relative=\"false\"/>\n");
                break;
            case FILTERJURECZKO:
                configFile.append(" <loader name=\"CSVFolderLoader\" datalocation=\"benchmark/data/JURECZKO\" relative=\"false\"/>\n");
                configFile.append(" <versionfilter name=\"MinInstanceNumberFilter\" param=\"100\" />\n");
                configFile.append(" <versionfilter name=\"UnbalancedFilter\" param=\"0.05\" />\n");
                break;
            case AEEEM:
                configFile.append(" <loader name=\"ARFFFolderLoader\" datalocation=\"benchmark/data/AEEEM\" relative=\"false\"/>\n");
                break;
            case AEEEM_LDHH:
                configFile.append(" <loader name=\"ARFFFolderLoader\" datalocation=\"benchmark/data/AEEEM_LDHH\" relative=\"false\"/>\n");
                break;
            case AEEEM_LDHHWCHU:
                configFile.append(" <loader name=\"ARFFFolderLoader\" datalocation=\"benchmark/data/AEEEM_LDHHWCHU\" relative=\"false\"/>\n");
                break;
            case AEEEM_WCHU:
                configFile.append(" <loader name=\"ARFFFolderLoader\" datalocation=\"benchmark/data/AEEEM_WCHU\" relative=\"false\"/>\n");
                break;
            case RELINK:
                configFile.append(" <loader name=\"RelinkFolderLoader\" datalocation=\"benchmark/data/RELINK\" relative=\"false\"/>\n");
                break;
            case NETGENE:
                configFile.append(" <loader name=\"NetgeneFolderLoader\" datalocation=\"benchmark/data/NETGENE\" relative=\"false\"/>\n");
                break;
            case SELECTEDJURECZKO:
                configFile.append(" <loader name=\"CSVFolderLoader\" datalocation=\"benchmark/data/SELECTEDJURECZKO\" relative=\"false\"/>\n");
                break;
            case SMARTSHARK_ALL:
                configFile.append(" <loader name=\"JsonFolderLoader\" datalocation=\"exp-smartshark/data\" relative=\"false\"/>\n");
                configFile.append(" <versionfilter name=\"MinInstanceNumberFilter\" param=\"100\" />\n");
                configFile.append(" <versionfilter name=\"UnbalancedFilter\" param=\"0.05\" />\n");
                break;
            case SMARTSHARK_AST:
                configFile.append(" <loader name=\"JsonFolderLoader\" datalocation=\"exp-smartshark/data\" relative=\"false\"/>\n");
                configFile.append(" <versionfilter name=\"MinInstanceNumberFilter\" param=\"100\" />\n");
                configFile.append(" <versionfilter name=\"UnbalancedFilter\" param=\"0.05\" />\n");
                configFile.append(" <setwisepreprocessor name=\"AttributeRemoval\" param=\"PDA LLOC PUA LOC McCC CLOC TNLM CLLC CCO TNPA NA AD NLPA NLS LDC NM TNPM LCOM5 WMC NOD RFC TNM NL NS NPA NOC CBO TNC TLLOC CI TNLG NLM NLG TNA DIT TCD TNLA NLE NG NLA TNLPA NOS CBOI NLPM LLDC CD TNG NPM CCL NOI NOP TLOC CLC CC DLOC NII TCLOC TNLS NOA TNLPM\"/>\n");
                break;
            case SMARTSHARK_SM:
                configFile.append(" <loader name=\"JsonFolderLoader\" datalocation=\"exp-smartshark/data\" relative=\"false\"/>\n");
                configFile.append(" <versionfilter name=\"MinInstanceNumberFilter\" param=\"100\" />\n");
                configFile.append(" <versionfilter name=\"UnbalancedFilter\" param=\"0.05\" />\n");
                configFile.append(" <setwisepreprocessor name=\"AttributeRemoval\" param=\"ReferenceType LambdaExpression Member TypeArgument ThrowStatement ArraySelector Declaration ClassCreator ForStatement SwitchStatement InnerClassCreator Literal TypeParameter VoidClassReference WhileStatement EnhancedForControl This Statement ForControl BinaryOperation MethodReference SuperMemberReference EnumBody FormalParameter EnumConstantDeclaration Expression PackageDeclaration VariableDeclarator AssertStatement Documented node_count DoStatement InterfaceDeclaration ReturnStatement Cast ExplicitConstructorInvocation EnumDeclaration SynchronizedStatement AnnotationMethod SwitchStatementCase MemberReference TypeDeclaration ArrayInitializer CatchClauseParameter CatchClause VariableDeclaration TryStatement Annotation TryResource MethodInvocation BasicType ElementArrayValue InferredFormalParameter IfStatement SuperConstructorInvocation BreakStatement AnnotationDeclaration FieldDeclaration Assignment ContinueStatement Import Primary BlockStatement ClassDeclaration TernaryExpression ClassReference CompilationUnit ConstantDeclaration LocalVariableDeclaration MethodDeclaration ConstructorDeclaration ElementValuePair ArrayCreator Invocation StatementExpression SuperMethodInvocation\"/>\n");
                break;
            default:
                throw new InvalidParameterException("Unknown data set: " + dataset.toString());
        }
        configFile.append(" <versionfilter name=\"MinClassNumberFilter\" param=\"5\" />\n");
        configFile.append(" <resultspath path=\"benchmark/results-csv\"/>\n");
    }
    
    public static String ALL(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainers(configFile);
        
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String CV(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainers(configFile);
        
        configFile.append(" <eval name=\"CVWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Random(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        
        configFile.append(" <trainer name=\"WekaTraining\" param=\"RANDOM de.ugoe.cs.cpdp.wekaclassifier.RandomClass\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        configFile.append(" <repetitions number=\"10\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Trivial(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        
        configFile.append(" <trainer name=\"WekaTraining\" param=\"FIX de.ugoe.cs.cpdp.wekaclassifier.FixClass -C 1\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Koshgoftaar08(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainersBagging(configFile);
        
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Watanabe08(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainers(configFile);
        
        configFile.append(" <setwisepreprocessor name=\"AverageStandardization\" param=\"\" />\n");        
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Turhan09(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainers(configFile);
        
        configFile.append(" <preprocessor name=\"LogarithmTransform\" param=\"\" />\n");
        configFile.append(" <pointwiseselector name=\"TurhanFilter\" param=\"10\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Zimmermann09(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainers(configFile);
        
        configFile.append(" <setwiseselector name=\"DecisionTreeSelection\" param=\"max median stddev\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String CamargoCruz09(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainers(configFile);
        
        configFile.append(" <preprocessor name=\"LogarithmTransform\" param=\"\" />\n");
        configFile.append(" <preprocessor name=\"MedianAsReference\" param=\"10\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Liu10(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        
        configFile.append(" <setwisetrainer name=\"GPTraining\" param=\"numberRuns:1,errorType2Weight:15\" />");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        configFile.append(" <repetitions number=\"10\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Menzies11(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        
        trainersLocalWhere(configFile);        
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        configFile.append(" <repetitions number=\"10\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Ma12(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainers(configFile);
        
        configFile.append(" <preprocessor name=\"DataGravitation\" param=\"\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Peters12(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainers(configFile);
        
        configFile.append(" <preprocessor name=\"MORPH\" param=\"\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        configFile.append(" <repetitions number=\"10\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Uchigaki12(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        
        configFile.append(" <preprocessor name=\"ZScoreNormalization\" param=\"\" />\n");
        configFile.append(" <trainer name=\"WekaTraining\" param=\"LE de.ugoe.cs.cpdp.wekaclassifier.LogisticEnsemble\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Canfora13(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        
        configFile.append(" <preprocessor name=\"ZScoreNormalization\" param=\"\" />\n");
        configFile.append(" <trainer name=\"WekaTraining\" param=\"MODEP de.ugoe.cs.cpdp.wekaclassifier.MODEPClassifier -R 0.7\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        configFile.append(" <repetitions number=\"10\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Peters13(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainers(configFile);
        
        configFile.append(" <preprocessor name=\"MORPH\" param=\"\" />\n");
        configFile.append(" <pointwiseselector name=\"CLIFF\" param=\"0.40\" />");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        configFile.append(" <repetitions number=\"10\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Herbold13(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainers(configFile);
        
        int numNeighbors;
        switch (dataset)
        {
            case AEEEM:
            case AEEEM_LDHH:
            case AEEEM_WCHU:
            case AEEEM_LDHHWCHU:
                numNeighbors = 2;
                break;
            case MDP:
                numNeighbors = 5;
                break;
            case JURECZKO:
                numNeighbors = 30;
                break;
            case FILTERJURECZKO:
                numNeighbors = 20;
                break;
            case RELINK:
                numNeighbors = 1;
                break;
            case NETGENE:
                numNeighbors = 1;
                break;
            case SELECTEDJURECZKO:
                numNeighbors = 4;
                break;
            case SMARTSHARK_ALL:
            case SMARTSHARK_AST:
            case SMARTSHARK_SM:
                // TODO check num neighbors
            default:
                numNeighbors = 10;
                break;
        }
        
        configFile.append(" <setwisepreprocessor name=\"Normalization\" param=\"\" />\n");
        configFile.append(" <setwiseselector name=\"SetWiseKNNSelection\" param=\""+ numNeighbors +"\" />\n");
        configFile.append(" <postprocessor name=\"BiasedWeights\" param=\"0.5\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String ZHe13(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainersBagging(configFile);
        
        int numNeighbors;
        switch (dataset)
        {
            case AEEEM:
            case AEEEM_LDHH:
            case AEEEM_LDHHWCHU:
            case AEEEM_WCHU:
                numNeighbors = 1;
                break;
            case MDP:
                numNeighbors = 4;
                break;
            case JURECZKO:
                numNeighbors = 16;
                break;
            case FILTERJURECZKO:
                numNeighbors = 13;
                break;
            case RELINK:
                numNeighbors = 1;
                break;
            case NETGENE:
                numNeighbors = 1;
                break;
            case SELECTEDJURECZKO:
                numNeighbors = 4;
                break;
            case SMARTSHARK_ALL:
            case SMARTSHARK_AST:
            case SMARTSHARK_SM:
                // TODO check num neighbors
            default:
                numNeighbors = 10;
                break;
        }
        
        configFile.append(" <setwisepreprocessor name=\"Normalization\" param=\"\" />\n");
        configFile.append(" <setwiseselector name=\"SeparatabilitySelection\" param=\"" + numNeighbors + "\" />\n");
        configFile.append(" <setwisepostprocessor name=\"Undersampling\" param=\"\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        configFile.append(" <repetitions number=\"10\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Nam13(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainers(configFile);
        
        configFile.append(" <preprocessor name=\"TCAPlusNormalization\" param=\"\" />\n");
        configFile.append(" <postprocessor name=\"TransferComponentAnalysis\" param=\"\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Panichella14(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        
        configFile.append(" <trainer name=\"WekaTraining\" param=\"CODEP-LR de.ugoe.cs.cpdp.wekaclassifier.LogisticCODEP\" />\n");
        configFile.append(" <trainer name=\"WekaTraining\" param=\"CODEP-BN de.ugoe.cs.cpdp.wekaclassifier.BayesNetCODEP\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Ryu14(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        
        configFile.append(" <preprocessor name=\"ZScoreNormalization\" param=\"\" />\n");
        configFile.append(" <testawaretrainer name=\"WekaTestAwareTraining\" param=\"VCBSVM de.ugoe.cs.cpdp.wekaclassifier.VCBSVM -L 0.1 -B 10\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        configFile.append(" <repetitions number=\"10\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String PHe15(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainers(configFile);
        
        configFile.append(" <setwisepreprocessor name=\"LogarithmTransform\" param=\"\" />\n");
        configFile.append(" <setwisepreprocessor name=\"TopMetricFilter\" param=\"\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Peters15(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainers(configFile);
        
        configFile.append(" <setwisepreprocessor name=\"LogarithmTransform\" param=\"\" />\n");
        configFile.append(" <setwiseselector name=\"LACE2\" param=\"0.4\" />\n");
        configFile.append(" <pointwiseselector name=\"TurhanFilter\" param=\"1\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        configFile.append(" <repetitions number=\"10\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Kawata15(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainers(configFile);
        
        configFile.append(" <pointwiseselector name=\"DBSCANFilter\" param=\"\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String YZhang15(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        
        configFile.append(" <trainer name=\"WekaTraining\" param=\"AVGVOTE weka.classifiers.meta.Vote -S 1 -B &quot;weka.classifiers.trees.ADTree&quot; -B &quot;de.ugoe.cs.cpdp.wekaclassifier.DecisionTableWrapper&quot; -B &quot;de.ugoe.cs.cpdp.wekaclassifier.BayesNetWrapper&quot; -B &quot;weka.classifiers.functions.MultilayerPerceptron&quot; -B &quot;weka.classifiers.functions.RBFNetwork&quot; -R AVG\" />\n");
        configFile.append(" <trainer name=\"WekaTraining\" param=\"MAXVOTE weka.classifiers.meta.Vote -S 1 -B &quot;weka.classifiers.trees.ADTree&quot; -B &quot;de.ugoe.cs.cpdp.wekaclassifier.DecisionTableWrapper&quot; -B &quot;de.ugoe.cs.cpdp.wekaclassifier.BayesNetWrapper&quot; -B &quot;weka.classifiers.functions.MultilayerPerceptron&quot; -B &quot;weka.classifiers.functions.RBFNetwork&quot; -R MAX\" />\n");
        configFile.append(" <trainer name=\"WekaTraining\" param=\"BAG-DT weka.classifiers.meta.Bagging -P 100 -S 1 -I 10 -W weka.classifiers.trees.J48\" />\n");
        configFile.append(" <trainer name=\"WekaTraining\" param=\"BAG-NB weka.classifiers.meta.Bagging -P 100 -S 1 -I 10 -W weka.classifiers.bayes.NaiveBayes\" />\n");
        configFile.append(" <trainer name=\"WekaTraining\" param=\"BOOST-DT weka.classifiers.meta.AdaBoostM1 -P 100 -S 1 -I 10 -W weka.classifiers.trees.J48\" />\n");
        configFile.append(" <trainer name=\"WekaTraining\" param=\"BOOST-NB weka.classifiers.meta.AdaBoostM1 -P 100 -S 1 -I 10 -W weka.classifiers.bayes.NaiveBayes\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Amasaki15(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainers(configFile);
        
        configFile.append(" <preprocessor name=\"LogarithmTransform\" param=\"\" />\n");
        configFile.append(" <preprocessor name=\"SynonymAttributePruning\" param=\"\" />\n");
        configFile.append(" <pointwiseselector name=\"SynonymOutlierRemoval\" param=\"\" />");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Ryu15(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainersLASER(configFile);
        
        configFile.append(" <pointwiseselector name=\"MahalanobisOutlierRemoval\" param=\"\" />\n");
        configFile.append(" <pointwiseselector name=\"NeighborhoodFilter\" param=\"\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
    
    public static String Nam15(Dataset dataset) {
        StringBuilder configFile = new StringBuilder();
        preamble(configFile);
        dataset(configFile, dataset);
        trainers(configFile);
        
        configFile.append(" <preprocessor name=\"CLAMIProcessor\" param=\"\" />\n");
        configFile.append(" <eval name=\"NormalWekaEvaluation\" param=\"\" />\n");
        
        postamble(configFile);
        return configFile.toString();
    }
}
